import { DEFAULT_REGIONS } from '../utils/regions.js';
import { isOpera } from '../utils/browser-info.js';
import Config, { FLAG_PAUSE_ASSISTANT, ACTION_PAUSE_ASSISTANT } from './config.js';
import CustomFilters from './custom-filters.js';
import ManagedConfig from './managed-config.js';
import store from '../npm/hybrids/src/store.js';

const UPDATE_OPTIONS_ACTION_NAME = "updateOptions";
const GLOBAL_PAUSE_ID = "<all_urls>";
const SYNC_OPTIONS = [
  "blockAds",
  "blockTrackers",
  "blockAnnoyances",
  "regionalFilters",
  "customFilters",
  "experimentalFilters",
  "serpTrackingPrevention",
  "wtmSerpReport",
  "trackerWheel",
  "trackerCount",
  "pauseAssistant",
  "panel",
  "theme"
];
const REPORT_OPTIONS = [...SYNC_OPTIONS, "filtersUpdatedAt"];
const SYNC_PROTECTED_OPTIONS = [...SYNC_OPTIONS, "exceptions", "paused"];
const ENGINES = [
  { name: "ads", key: "blockAds" },
  { name: "tracking", key: "blockTrackers" },
  { name: "annoyances", key: "blockAnnoyances" }
];
const OPTIONS_VERSION = 3;
const Options = {
  // Main features
  blockAds: true,
  blockTrackers: true,
  blockAnnoyances: true,
  // Regional filters
  regionalFilters: {
    enabled: DEFAULT_REGIONS.length > 0,
    regions: DEFAULT_REGIONS.length ? DEFAULT_REGIONS : [String]
  },
  // Advanced features
  customFilters: {
    enabled: false,
    trustedScriptlets: false
  },
  experimentalFilters: false,
  filtersUpdatedAt: 0,
  serpTrackingPrevention: true,
  // WhoTracks.Me Section
  wtmSerpReport: true,
  trackerWheel: false,
  ...{ trackerCount: true } ,
  pauseAssistant: true,
  // Onboarding
  terms: false,
  feedback: true,
  onboarding: {
    shown: 0,
    ...isOpera() ? { serpShownAt: 0, serpShown: 0 } : {},
    ...{ pinIt: false } 
  },
  // UI
  panel: { statsType: "graph", notifications: true },
  theme: "",
  // Tracker exceptions
  exceptions: store.record({ global: false, domains: [String] }),
  // Paused domains
  paused: store.record({ revokeAt: 0, assist: false }),
  // Sync
  sync: true,
  revision: 0,
  // What's new
  whatsNewVersion: 0,
  [store.connect]: {
    async get() {
      let { options = {}, optionsVersion } = await chrome.storage.local.get([
        "options",
        "optionsVersion"
      ]);
      if (!optionsVersion) {
        chrome.storage.local.set({ optionsVersion: OPTIONS_VERSION });
      } else if (optionsVersion < OPTIONS_VERSION) {
        await migrate(options, optionsVersion);
      }
      {
        return manage(options);
      }
    },
    async set(_, options) {
      options = options || {};
      await chrome.storage.local.set({
        options: (
          // Firefox does not serialize correctly objects with getters
          options
        )
      });
      chrome.runtime.sendMessage({
        action: UPDATE_OPTIONS_ACTION_NAME
      }).catch(() => {
      });
      return options;
    }
  }
};
chrome.runtime.onMessage.addListener((msg) => {
  if (msg.action === UPDATE_OPTIONS_ACTION_NAME) {
    store.clear(Options, false);
    store.get(Options);
  }
});
async function migrate(options, optionsVersion) {
  if (optionsVersion < 2) {
    if (options.paused) {
      options.paused = options.paused.reduce((acc, { id, revokeAt }) => {
        acc[id] = { revokeAt };
        return acc;
      }, {});
    }
    console.debug("[options] Migrated to version 2:", options);
  }
  if (optionsVersion < 3) {
    const { text } = await store.resolve(CustomFilters);
    if (text) {
      options.customFilters = {
        ...options.customFilters,
        enabled: true
      };
    }
    console.debug("[options] Migrated to version 3:", options);
  }
  await chrome.storage.local.set({
    options,
    optionsVersion: OPTIONS_VERSION
  });
}
async function manage(options) {
  const managed = await store.resolve(ManagedConfig);
  if (managed.disableOnboarding === true) {
    options.terms = true;
    options.onboarding = { shown: 1 };
  }
  if (managed.disableUserControl === true) {
    options.sync = false;
    options.paused = {};
    const config = await store.resolve(Config);
    if (config.hasFlag(FLAG_PAUSE_ASSISTANT)) {
      for (const [domain, { actions }] of Object.entries(config.domains)) {
        if (actions.includes(ACTION_PAUSE_ASSISTANT)) {
          options.paused[domain] = { revokeAt: 0, assist: true };
        }
      }
    }
  }
  managed.trustedDomains.forEach((domain) => {
    options.paused ||= {};
    options.paused[domain] = { revokeAt: 0 };
  });
  return options;
}
function getPausedDetails(options, hostname = "") {
  if (options.paused[GLOBAL_PAUSE_ID]) {
    return options.paused[GLOBAL_PAUSE_ID];
  }
  if (!hostname) return null;
  const pausedHostname = Object.keys(options.paused).find(
    (domain) => hostname.endsWith(domain)
  );
  return pausedHostname ? options.paused[pausedHostname] : null;
}

export { ENGINES, GLOBAL_PAUSE_ID, REPORT_OPTIONS, SYNC_OPTIONS, SYNC_PROTECTED_OPTIONS, Options as default, getPausedDetails };
